import sys
sys.path.append('./TM_Adapter_final/dataset')
import os
import argparse
import joblib
import lightning.pytorch as pl
import torch.distributed as dist
from pytorch_lightning.loggers import CSVLogger
from lightning.pytorch.callbacks import ModelCheckpoint

from util import *

from util_tools.optim_factory import LayerDecayValueAssigner
from dataset.datasets import build_dataset
from util_tools.mixup import Mixup
from util_tools.utils import cross_multiple_samples_collate
# torch.autograd.set_detect_anomaly(True)
# os.environ["TORCH_DISTRIBUTED_DEBUG"] = "INFO"

def main():
    parser = argparse.ArgumentParser('TM-Adapter final', add_help=False)
    
    # Directory name
    parser.add_argument('--project_name', type=str)

    # Training setting
    parser.add_argument('--batch_size', default=64, type=int)
    parser.add_argument('--epochs', default=30, type=int)

    # Model parameters
    parser.add_argument('--model', default='ViT-B/16', type=str, metavar='MODEL',
                        help='Name of model to train')
    parser.add_argument('--input_size', default=224, type=int,
                        help='videos input size')
    parser.add_argument('--merge_token', default=4, type=int,
                        help='Number of merged frame in Global attention modeling')
    parser.add_argument('--block_layers', default=29, type=int,
                        help='Global adapter Blocks layer')
    parser.add_argument('--additional_local_adpater', default=False, type=bool,
                        help='Define additional Local Adapter')
    parser.add_argument('--concatenation_rule', type=str,
                        help='Define concatenation rule of Local adpater and Global adapter')
    parser.add_argument('--pre_trained_dataset', type=str, default = 'WIT',
                        help='Pre-trained dataset WIT, LAION, DFN2B')
                    

    parser.add_argument('--fast_dev_run', action='store_true', default=False, help='True: Training without error')
    parser.add_argument('--profile', action='store_true', default=False, help='True: Training without error')

    # Optimizer parameters
    parser.add_argument('--opt', default='adamw', type=str, metavar='OPTIMIZER',
                        help='Optimizer (default: "adamw"')
    parser.add_argument('--opt_eps', default=1e-8, type=float, metavar='EPSILON',
                        help='Optimizer Epsilon (default: 1e-8)')
    parser.add_argument('--opt_betas', default=None, type=float, nargs='+', metavar='BETA',
                        help='Optimizer Betas (default: None, use opt default)')
    parser.add_argument('--clip_grad', type=float, default=None, metavar='NORM',
                        help='Clip gradient norm (default: None, no clipping)')
    parser.add_argument('--momentum', type=float, default=0.9, metavar='M',
                        help='SGD momentum (default: 0.9)')
    parser.add_argument('--weight_decay', type=float, default=0.05,
                        help='weight decay (default: 0.05)')
    parser.add_argument('--weight_decay_end', type=float, default=None, help="""Final value of the
        weight decay. We use a cosine schedule for WD and using a larger decay by
        the end of training improves performance for ViTs.""")

    parser.add_argument('--lr', type=float, default=1e-3, metavar='LR',
                        help='learning rate (default: 1e-3)')
    parser.add_argument('--layer_decay', type=float, default=0.75)

    parser.add_argument('--warmup_lr', type=float, default=1e-6, metavar='LR',
                        help='warmup learning rate (default: 1e-6)')
    parser.add_argument('--min_lr', type=float, default=1e-6, metavar='LR',
                        help='lower lr bound for cyclic schedulers that hit 0 (1e-5)')

    parser.add_argument('--warmup_epochs', type=int, default=5, metavar='N',
                        help='epochs to warmup LR, if scheduler supports')
    parser.add_argument('--warmup_steps', type=int, default=-1, metavar='N',
                        help='num of steps to warmup LR, will overload warmup_epochs if set > 0')

    # Augmentation parameters
    parser.add_argument('--color_jitter', type=float, default=0.4, metavar='PCT',
                        help='Color jitter factor (default: 0.4)')
    parser.add_argument('--num_sample', type=int, default=2,
                        help='Repeated_aug (default: 2)')
    parser.add_argument('--aa', type=str, default='rand-m7-n4-mstd0.5-inc1', metavar='NAME',
                        help='Use AutoAugment policy. "v0" or "original". " + "(default: rand-m7-n4-mstd0.5-inc1)'),
    parser.add_argument('--smoothing', type=float, default=0.1,
                        help='Label smoothing (default: 0.1)')
    parser.add_argument('--train_interpolation', type=str, default='bicubic',
                        help='Training interpolation (random, bilinear, bicubic default: "bicubic")')

    # Evaluation parameters
    parser.add_argument('--crop_pct', type=float, default=None)
    parser.add_argument('--short_side_size', type=int, default=224)
    parser.add_argument('--test_num_segment', type=int, default=5)
    parser.add_argument('--test_num_crop', type=int, default=3)
    
    # Random Erase params
    parser.add_argument('--rand_erase', type=bool, default=False),
    parser.add_argument('--reprob', type=float, default=0.25, metavar='PCT',
                        help='Random erase prob (default: 0.25)')
    parser.add_argument('--remode', type=str, default='pixel',
                        help='Random erase mode (default: "pixel")')
    parser.add_argument('--recount', type=int, default=1,
                        help='Random erase count (default: 1)')
    parser.add_argument('--resplit', action='store_true', default=False,
                        help='Do not random erase first (clean) augmentation split')

    # Mixup params
    parser.add_argument('--mixup', type=float, default=0.8,
                        help='mixup alpha, mixup enabled if > 0.')
    parser.add_argument('--cutmix', type=float, default=1.0,
                        help='cutmix alpha, cutmix enabled if > 0.')
    parser.add_argument('--cutmix_minmax', type=float, nargs='+', default=None,
                        help='cutmix min/max ratio, overrides alpha and enables cutmix if set (default: None)')
    parser.add_argument('--mixup_prob', type=float, default=1.0,
                        help='Probability of performing mixup or cutmix when either/both is enabled')
    parser.add_argument('--mixup_switch_prob', type=float, default=0.5,
                        help='Probability of switching to cutmix when both mixup and cutmix enabled')
    parser.add_argument('--mixup_mode', type=str, default='batch',
                        help='How to apply mixup/cutmix params. Per "batch", "pair", or "elem"')

    # Dataset parameters
    parser.add_argument('--data_path', default='/path/to/list_kinetics-400', type=str,
                        help='dataset path')
    parser.add_argument('--anno_path', default=None, type=str, help='annotation path')
    parser.add_argument('--nb_classes', default=400, type=int,
                        help='number of the classification types')

    parser.add_argument('--num_segments', type=int, default= 1)
    parser.add_argument('--num_frames', type=int, default= 16)
    parser.add_argument('--sampling_rate', type=int, default= 4)
    parser.add_argument('--data_set', default='Kinetics-400', #choices=['Kinetics-400', 'SSV2', 'SSV1', 'UCF101', 'HMDB51','image_folder'],
                        type=str, help='dataset')
    parser.add_argument('--device', default=None, type=int,
                        help='device to use for training / testing')
    parser.add_argument('--seed', default=0, type=int)

    parser.add_argument('--eval', action='store_true',
                        help='Perform evaluation only')
    parser.add_argument('--num_workers', default=10, type=int)
    parser.add_argument('--pin_mem', action='store_true',
                        help='Pin CPU memory in DataLoader for more efficient (sometimes) transfer to GPU.')
    parser.add_argument('--no_pin_mem', action='store_false', dest='pin_mem')

    parser.set_defaults(pin_mem=True)
    args = parser.parse_args()
    
    if args.device == None:
        args.device = [0, 1, 2, 3]
    elif args.device == 21:
        args.device = [0, 1]
    elif args.device == 31:
        args.device = [0, 1, 2]
    else:
        args.device = [args.device]

    if args.additional_local_adpater == True:
        additional_adapter = '24LA'
    else:
        additional_adapter = '12LA'
    os.environ["WANDB_CACHE_DIR"] = f"./trained_model/cache/"


    save_dir = f'./{args.project_name}/{args.data_set}_{args.num_frames}/{args.block_layers}_{args.merge_token}_{additional_adapter}/{args.concatenation_rule}/csvlog/'
    
    if not os.path.exists(save_dir):
        os.makedirs(save_dir)
    
    print('save_dir:    ', save_dir)


    Block_layers = get_block_layer(args.block_layers)

    print(f'-----------------------------------------------------------------Creating dataset-----------------------------------------------------------------')
    dataset_train, args.nb_classes = build_dataset(is_train=True, test_mode=False, args=args)
    dataset_val, _ = build_dataset(is_train=False, test_mode=False, args=args)
    
    if args.num_sample > 1:
        collate_func = partial(cross_multiple_samples_collate, fold=False)
    else:
        collate_func = None
    
    print('args.pin_mem:        ', args.pin_mem)
    dataloader_train = torch.utils.data.DataLoader(
        dataset_train, 
        batch_size=args.batch_size,
        num_workers=args.num_workers,
        pin_memory=args.pin_mem,
        drop_last=True,
        collate_fn=collate_func,
        persistent_workers=True
    )

    dataloader_val = torch.utils.data.DataLoader(
            dataset_val,
            batch_size=args.batch_size,
            num_workers=args.num_workers,
            pin_memory=args.pin_mem,
            persistent_workers=True
        )

    CSV_logger = CSVLogger(save_dir,
    flush_logs_every_n_steps = len(dataloader_train)//4)
    saved_path = f"{save_dir}"

    mixup_fn = None
    mixup_active = args.mixup > 0 or args.cutmix > 0. or args.cutmix_minmax is not None
    if mixup_active:
        print("Mixup is activated!")
        mixup_fn = Mixup(
            mixup_alpha=args.mixup, cutmix_alpha=args.cutmix, cutmix_minmax=args.cutmix_minmax,
            prob=args.mixup_prob, switch_prob=args.mixup_switch_prob, mode=args.mixup_mode,
            label_smoothing=args.smoothing, num_classes=args.nb_classes)
    
    print('args.profile:        ', args.profile)
    if args.profile:
        from lightning.pytorch.profilers import AdvancedProfiler
        profiler = AdvancedProfiler(dirpath=".", filename="perf_logs")
    else:
        profiler = None

    checkpoint_callback = ModelCheckpoint(
        save_top_k=5,
        save_last = True,
        monitor="val_loss",
        mode="min",
        dirpath=saved_path,
        filename=f"best_model",
    )
    print('device:          ', args.device)
    print('profiler:        ', profiler)
    trainer = pl.Trainer(
        max_epochs=args.epochs,
        callbacks = [checkpoint_callback],
        fast_dev_run=args.fast_dev_run,
        logger=[CSV_logger],
        accelerator="gpu",
        devices = args.device,
        strategy='deepspeed_stage_2',
        precision = '16-mixed',
        profiler=profiler,
        log_every_n_steps = len(dataloader_train)//4)
    if trainer.is_global_zero:
        print(f'-----------------------------------------------------------------Creating model-----------------------------------------------------------------')

    if args.model == 'ViT-B/16':
        from backbone.clip_B16 import build_clip
        model = build_clip(num_classes = args.nb_classes, merge_token = args.merge_token, Block_layers = Block_layers, additional_local_adpater= args.additional_local_adpater, concatenation_rule = args.concatenation_rule, pre_trained_dataset = args.pre_trained_dataset)
    elif args.model == 'ViT-L/14':
        from backbone.clip_L14 import build_clip
        model = build_clip(num_classes = args.nb_classes, merge_token = args.merge_token, Block_layers = Block_layers, additional_local_adpater= args.additional_local_adpater, concatenation_rule = args.concatenation_rule, pre_trained_dataset = args.pre_trained_dataset)
    
    if trainer.is_global_zero:
        print(model)
    args.lr =  args.lr * args.batch_size * len(args.device) / 256
    args.min_lr = args.min_lr * args.batch_size * len(args.device) / 256
    args.warmup_lr = args.warmup_lr * args.batch_size * len(args.device) / 256
    
    if args.layer_decay < 1.0:
        if args.model == "ViT-B/16" or args.model == "ViT-B/32":
            num_layers = 12
        elif args.model == "ViT-L/14" or args.model == "ViT-L/14-336px" or args.model == "EVA02-CLIP-L-14" or args.model == "EVA02-CLIP-L-14-336":
            num_layers = 24
        elif args.model == "EVA02-CLIP-bigE-14":
            num_layers = 64
        else:
            print("only support CLIP ViT-B/16, ViT-B/32, ViT-L/14, ViT-L/14-336px, EVA02-CLIP-L-14, EVA02-CLIP-L-14-336 and EVA02-CLIP-bigE-14 for laryer wise decay")
            exit(0)

        assigner = LayerDecayValueAssigner(list(args.layer_decay ** (num_layers + 1 - i) for i in range(num_layers + 2)))
    else:
        assigner = None

    if assigner is not None:
        print("Assigned values = %s" % str(assigner.values))
    
    model = lightening_module(model,
            num_classes = args.nb_classes,
            mixup_fn = mixup_fn,
            assigner = assigner,
            len_train = len(dataloader_train),
            save_directory = saved_path,
            args = args)

    
    # trainer.fit(model, train_dataloaders=dataloader_train, val_dataloaders=dataloader_val)
    # torch.distributed.barrier()
    if trainer.is_global_zero:
        print(f'-----------------------------------------------------------------Testing model-----------------------------------------------------------------')
    
    ### Final test
    dataset_test, _ = build_dataset(is_train=False, test_mode=True, args=args)
    dataloader_test = torch.utils.data.DataLoader(
            dataset_test,
            batch_size=args.batch_size,
            num_workers=args.num_workers,
            pin_memory=args.pin_mem,
        )

    
    for path in os.listdir(saved_path):
        if not path.endswith('ckpt'):
            continue
        model_path = saved_path + '/' + path
        
        result = trainer.predict(model, dataloaders=dataloader_test, ckpt_path=model_path)

        gpu_rank = trainer.local_rank

        if not os.path.exists(f'{saved_path}/temp'):
            os.makedirs(f'{saved_path}/temp')

        joblib.dump(pd.concat(result, ignore_index=True), f'{saved_path}/temp/test_result_{gpu_rank}.joblib')

        torch.distributed.barrier()
        
        if trainer.is_global_zero:
            final_data = pd.DataFrame()
            for rank in args.device:
                loaded_data = joblib.load(f'{saved_path}/temp/test_result_{rank}.joblib')
                # data DataFrame이 비어있는 경우
                if final_data.empty:
                    final_data = loaded_data
                else:
                    # data DataFrame에 로드된 데이터 추가
                    final_data = pd.concat([final_data, loaded_data], ignore_index=True)
            final_data = final_data.reset_index(drop=True)
            
            merge_final_result(final_data, args, saved_path, path)
if __name__ == '__main__': main()
